<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class SystemDiagnosticsController extends Controller
{
    /**
     * Show log files and their content (tail).
     */
    public function logs(Request $request)
    {
        $logPath = storage_path('logs');

        // All .log files sorted by mtime desc
        $files = collect(File::files($logPath))
            ->filter(function ($file) {
                return $file->getExtension() === 'log';
            })
            ->sortByDesc(function ($file) {
                return $file->getMTime();
            })
            ->map(function ($file) {
                return $file->getFilename();
            })
            ->values()
            ->all();

        $activeFile = $request->get('file', $files[0] ?? null);
        $content    = null;
        $fileSize   = null;
        $truncated  = false;
        $tailBytes  = 1024 * 1024 * 2; // 2 MB tail

        if ($activeFile && File::exists($logPath . DIRECTORY_SEPARATOR . $activeFile)) {
            $fullPath = $logPath . DIRECTORY_SEPARATOR . $activeFile;
            $fileSize = filesize($fullPath);

            [$content, $truncated] = $this->readTail($fullPath, $tailBytes);
        }

        return view('system.logs', [
            'files'      => $files,
            'activeFile' => $activeFile,
            'content'    => $content,
            'fileSize'   => $fileSize,
            'truncated'  => $truncated,
            'tailBytes'  => $tailBytes,
        ]);
    }

    /**
     * Read at most $maxBytes from the end of a file.
     * Returns [content, truncated(bool)].
     */
    protected function readTail(string $path, int $maxBytes): array
    {
        $size = filesize($path);

        // Small file: read everything
        if ($size <= $maxBytes) {
            return [File::get($path), false];
        }

        // Big file: read last $maxBytes bytes
        $fh = fopen($path, 'r');
        if (!$fh) {
            return ['Unable to open log file.', false];
        }

        $offset = $size - $maxBytes;
        fseek($fh, $offset);
        $data = fread($fh, $maxBytes);
        fclose($fh);

        // Remove first partial line
        $newlinePos = strpos($data, "\n");
        if ($newlinePos !== false) {
            $data = substr($data, $newlinePos + 1);
        }

        return [$data, true];
    }

    /**
     * Performance / memory analysis based on memprof.log.
     */
    public function analysis()
    {
        $logPath = storage_path('logs/memprof.log');
        $entries = [];

        if (File::exists($logPath)) {
            // Load lines (limit to last N to avoid huge file issues)
            $lines = file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: [];
            $maxLines = 5000; // last 5000 entries
            if (count($lines) > $maxLines) {
                $lines = array_slice($lines, -$maxLines);
            }

            foreach ($lines as $line) {
                $data = json_decode($line, true);
                if (is_array($data)) {
                    $entries[] = $data;
                }
            }
        }

        $totalEntries = count($entries);
        $statsByRoute = [];

        foreach ($entries as $row) {
            // Try to detect route / uri / label
            $route = $row['route'] ?? $row['uri'] ?? $row['path'] ?? $row['label'] ?? 'unknown';

            // Duration (ms) and memory (MB or similar) – adapt to your memprof format
            $duration = (float) ($row['duration_ms'] ?? $row['time_ms'] ?? $row['duration'] ?? 0);
            $memory   = (float) ($row['memory_mb'] ?? $row['memory'] ?? $row['memory_peak_mb'] ?? 0);
            $time     = $row['time'] ?? $row['timestamp'] ?? null;

            if (!isset($statsByRoute[$route])) {
                $statsByRoute[$route] = [
                    'route'          => $route,
                    'count'          => 0,
                    'total_duration' => 0.0,
                    'max_duration'   => 0.0,
                    'total_memory'   => 0.0,
                    'max_memory'     => 0.0,
                    'last_called_at' => $time,
                ];
            }

            $s = &$statsByRoute[$route];

            $s['count']++;
            $s['total_duration'] += $duration;
            $s['max_duration'] = max($s['max_duration'], $duration);
            $s['total_memory'] += $memory;
            $s['max_memory'] = max($s['max_memory'], $memory);

            if ($time && (!$s['last_called_at'] || $time > $s['last_called_at'])) {
                $s['last_called_at'] = $time;
            }

            unset($s);
        }

        // Compute averages & share percentage
        foreach ($statsByRoute as &$s) {
            $s['avg_duration'] = $s['count'] ? $s['total_duration'] / $s['count'] : 0;
            $s['avg_memory']   = $s['count'] ? $s['total_memory'] / $s['count'] : 0;
            $s['percentage']   = $totalEntries ? ($s['count'] / $totalEntries) * 100 : 0;
        }
        unset($s);

        // Sort clones for different views
        $byCount = array_values($statsByRoute);
        usort($byCount, fn ($a, $b) => $b['count'] <=> $a['count']);

        $byAvgDuration = array_values($statsByRoute);
        usort($byAvgDuration, fn ($a, $b) => $b['avg_duration'] <=> $a['avg_duration']);

        $summary = [
            'total_entries' => $totalEntries,
            'total_routes'  => count($statsByRoute),
            'most_called'   => $byCount[0]      ?? null,
            'slowest_route' => $byAvgDuration[0] ?? null,
        ];

        return view('system.analysis', [
            'summary'       => $summary,
            'byCount'       => $byCount,
            'byAvgDuration' => $byAvgDuration,
        ]);
    }
}
